#include <unistd.h>
#include <random>
#include <getopt.h>
#include <sys/wait.h>
#include <exception>
#include <iostream>
#include <fstream>
#include <iomanip>
#include <memory>
#include <vector>
#include <cstdio>
#include <cstring>
#include <cstdarg>
#include <cstdlib>
#include <cerrno>
using namespace std;
typedef long long ll;

int b, m, n=1;
ll seed;
bool help, verbose;
int instance_cnt = 4;

class spawn {
  int write_pipe[2];
  int read_pipe[2];
  int child_pid;
  FILE *out_file = NULL;
  public:
  spawn(char * const argv[], char *out_file_name) {
    if (pipe(write_pipe) == -1) throw runtime_error("Failed to create pipe");
    if (pipe(read_pipe) == -1) throw runtime_error("Failed to create pipe");
    child_pid = fork();
    if (child_pid == -1) throw runtime_error("Failed to fork process");
    if (child_pid == 0) { // in child
      if (dup2(write_pipe[0], STDIN_FILENO) == -1) throw runtime_error("Failed to dup handle");
      if (dup2(read_pipe[1], STDOUT_FILENO) == -1) throw runtime_error("Failed to dup handle");
      if (close(write_pipe[0]) == -1) throw runtime_error("Failed to close handle");
      if (close(write_pipe[1]) == -1) throw runtime_error("Failed to close handle");
      if (close(read_pipe[0]) == -1) throw runtime_error("Failed to close handle");
      if (close(read_pipe[1]) == -1) throw runtime_error("Failed to close handle");
      if (execv(argv[0], argv) == -1) {
        cerr << "Error: Failed to launch program\n";
        exit(1);
      }
    } else { // in parent
      if (close(write_pipe[0]) == -1) throw runtime_error("Failed to close handle");
      if (close(read_pipe[1]) == -1) throw runtime_error("Failed to close handle");
      signal(SIGPIPE, SIG_IGN); // 
      if(out_file_name){
        if(verbose)
          cerr <<"Opening file \"" << out_file_name << "\" for wrinting\n";
        out_file = fopen(out_file_name, "w");
      }
    }
  }

  bool send_line(const char *fmt, ...) {
    char buf[2000];
    va_list ap;
    va_start(ap, fmt);
    //int n = vsnprintf(0, 0, fmt, ap);
    //if (n < 0)  throw runtime_error("Invalid format");
    //if (n >= sizeof(buf)) throw runtime_error("Buffer overflow");
    int n = vsnprintf(buf, sizeof(buf), fmt, ap);
    va_end(ap);
    if(out_file){
      fputs(buf, out_file);
      fflush(out_file);
    }
    return write(write_pipe[1], buf, n) == n;
  }

  bool send_eof() {
    return close(write_pipe[1]) != -1;
  }

  bool recv_line(char buf[], int size) {
    for (int n = 0; n + 1 < size; ++n) {
      int m = read(read_pipe[0], buf + n, 1);
      if (m < 0) {
        return false;
      }
      if (m == 0) {
        buf[n] = 0;
        return true;
      }
      if (buf[n] == '\n') {
        buf[n] = 0;
        return true;
      }
    }
    buf[size - 1] = 0;
    return true;
  }

  int wait() {
    int status;
    waitpid(child_pid, &status, 0);
    return status;
  }
};

#define done(x, y) ( \
    cout << "Avg writes: " << (x) << '\n' << "Verdict: " << (y) << '\n')

mt19937 gen;

void gen_bits(char *res, int k) {
  for (int i = 0; i < k; ++i) res[i] = '0' + (gen() & 1);
}

char *out_file_name[10];

int main(int argc, char **argv) {
  {
    int c;
    bool err = false;
    while (1)
    {
      static struct option long_options[] =
      {
        /* These options don’t set a flag.
           We distinguish them by their indices. */
        {"mem",     required_argument, 0, 'b'},
        {"var",required_argument, 0, 'm'},
        {"cnt",     required_argument, 0, 'n'},
        {"help",    no_argument,       0, 'h'},
        {"usage",   no_argument,       0, 'h'},
        {"verbose", no_argument,       0, 'v'},
        {"instances", required_argument, 0, 'i'},
        {"seed",    required_argument, 0, 's'},
        {0, 0, 0, 0}
      };
      int option_index = 0;
      c = getopt_long (argc, argv, "b:m:n:hvi:s:0:1:2:3:4:5:6:7:8:9:",
          long_options, &option_index);
      /* Detect the end of the options. */
      if (c == -1)
        break;
      switch (c){
        case 'h':
          if(!help){
            const char *help_message = 
              "Usage: ./tester [options] prog\n"
              "Options:\n"
              "  -h   or --help          print this message\n"
              "required:\n"
              "  -b B or --mem B         memory length\n"
              "  -b M or --var M         variable length\n"
              "optional:\n"
              "  -n N or --cnt N         reset memory to zero N times (default is 1)\n"
              "  -v   or --verbose       print out all communication with instances of prog\n"
              "advanced:\n"
              "  -s N or --seed N        seed for generating random values (default is 0)\n"
              "  -i N or --instances N   start N (even) instances of prog (default is 4)\n"
              "    First half of these instances will be writers and second half will be readers\n"
              "  -N <file>               write input given to N-th instance into <file> (N should be in range 0..9)"
              "\n";
              puts(help_message);
          }
          help = true;
          break;
        case 'v':
          verbose = true; //+
          break;
        case 'b':
          b = atoi(optarg);//+
          break;
        case 'm':
          m = atoi(optarg);//+
          break;
        case 'n':
          n = atoi(optarg);//+
          break;
        case 'i':
          instance_cnt = atoi(optarg);//+
          if(!instance_cnt){
            printf ("%s: Number of instances has to be positive\n", argv[0]);
            err = true;
          }
          if(instance_cnt%2){
            printf ("%s: Number of instances has to be even\n", argv[0]);
            err = true;
          }
          break;
        case 's':
          seed = atoi(optarg);//+
          break;
        case '?':
          err = true;
          /* getopt_long already printed an error message. */
          break;
        default:
          out_file_name[c-'0'] = optarg;
      }
      if(err)
        return 0;
    }
    if(help){
      return 0;
    }
    /* Print any remaining command line arguments (not options). */
    if (optind >= argc){
      printf ("%s: No executable given\n", argv[0]);
      return 0;
    }
  }
  {
    bool err = false;
    if(!b){
      err = true;
      puts("No b(memory lenghth) given");
    }
    if(!m){
      err = true;
      puts("No m(message length) given");
    }
    if(err)
      return 0;
  }


  gen = mt19937(seed);
  gen.discard(gen.state_size);

  spawn **prog = new spawn *[instance_cnt];
  for (int i = 0; i < instance_cnt; ++i) {
    prog[i] = new spawn(argv + optind, out_file_name[i]);
  }

  bool good_com = true;
  for (int i = 0; i < instance_cnt; ++i) {
    if(verbose){
      cerr << i << " <- " << (i >= instance_cnt/2) << '\n';
      cerr << i << " <- "  << b << ' ' << m << '\n';
    }
    if (!prog[i]->send_line("%d\n%d %d\n", i >= instance_cnt/2, b, m)) {
      cerr << i << " initialization failed " << errno << '\n';
      done(0, "Presentation error");
      good_com = false;
    }
  }


  char *message = new char[m + 1];
  char *mem = new char[b + 1];
  char buffer[2000], junk[10];
  int write_cnt = 0;
  for (int ct = 0; ct < n && good_com; ++ct) {
    memset(mem, '0', b);
    mem[b] = 0;
    for (int i = 0; good_com; ++i) {
      gen_bits(message, m);
      message[m] = 0;

      int id = i%(instance_cnt/2);
      if (!prog[id]->send_line("%d\n%s %s\n", 1, mem, message)) {
        cerr << id << " sending write request failed " << errno << '\n';
        done(0, "Presentation error");
        good_com = false;
        break;
      }
      if(verbose){
        cerr << id << " <- 1"  << '\n';
        cerr << id << " <- " << mem << ' ' << message << '\n';
      }
      if (!prog[id]->recv_line(buffer, sizeof(buffer))) {
        cerr << id << " reading write status failed " << errno << '\n';
        done(0, "Presentation error");
        good_com = false;
        break;
      }
      int suc;
      if (sscanf(buffer, "%d%5s", &suc, junk) != 1 || (suc != 1 && suc != 0)) {
        if(verbose)
          cerr << id << " -> " << buffer << '\n';
        done(0, "Presentation error");
        good_com = false;
        break;
      }
      if (suc == 0) {
        if(verbose)
          cerr << id << " -> 0" << '\n';
        break;
      }
      if (!prog[id]->recv_line(buffer, sizeof(buffer))) {
        cerr << id << " reading write result failed " << errno << '\n';
        done(0, "Presentation error");
        good_com = false;
        break;
      }
      int end = strspn(buffer, "01");
      if (end != b || sscanf(buffer + end, "%5s", junk) != EOF) {
        if(verbose){
          cerr << id << " -> 1"  << '\n';
          cerr << id << " -> " << buffer << '\n';
        }
        done(0, "Presentation error");
        good_com = false;
        break;
      }
      buffer[end] = 0;
      if(verbose){
        cerr << id << " -> 1"  << '\n';
        cerr << id << " -> " << buffer << '\n';
      }
      for (int j = 0; j < b; ++j) {
        if (buffer[j] < mem[j]) {
          cerr << id << " writer decreased bit " << j << " of memory\n";
          done(0, "Invalid write");
          good_com = false;
          break;
        }
      }
      ++write_cnt;
      strcpy(mem, buffer);
      id += (instance_cnt/2);
      if (!prog[id]->send_line("%d\n%s\n", 1, mem)) {
        cerr << id << " sending read request failed " << errno << '\n';
        done(0, "Presentation error");
        good_com = false;
        break;
      }
      if(verbose){
        cerr << id << " <- 1" <<  '\n';
        cerr << id << " <- " << mem << '\n';
      }
      if (!prog[id]->recv_line(buffer, sizeof(buffer))) {
        cerr << id << " reading read result failed " << errno << '\n';
        done(0, "Presentation error");
        good_com = false;
        break;
      }
      end = strspn(buffer, "01");
      if (end != m || sscanf(buffer + end, "%5s", junk) != EOF) {
        if(verbose){
          cerr << id << " -> 1" <<  '\n';
          cerr << id << " -> " << buffer << '\n';
        }
        done(0, "Presentation error");
        good_com = false;
        break;
      }
      buffer[end] = 0;
      if(verbose)
        cerr << id << " -> " << buffer << '\n';
      for (int j = 0; j < m; ++j) {
        if (buffer[j] != message[j]) {
          cerr << id << " reader misread bit " << j << " of value\n";
          done(-1, "Incorrect read");
          good_com = false;
          break;
        }
      }
    }
  }
  for (int i = 0; i < instance_cnt; ++i) {
    if (!prog[i]->send_line("%d\n", 0)) {
      cerr << i << " shutdown failed " << errno << '\n';
      done(0, "Presentation error");
      good_com = false;
    }
    prog[i]->send_eof();
    if(verbose)
      cerr << i << " <- " << 0 << '\n';
  }
  int status = 0;
  for (int i = 0; i < instance_cnt; ++i) {
    int cur_status = prog[i]->wait();
    if (cur_status) {
      status = cur_status;
      cerr << i << " wait " << cur_status << '\n';
    }
  }
  if (good_com){
    long double res = (long double)write_cnt / n;
    done(res, "Output is correct");
  }

  delete[] mem;
  delete[] message;
  for (int i = 0; i < instance_cnt; ++i) {
    delete prog[i];
  }
  delete[] prog;
  return status; // pass child's status back to environment
}
